﻿# 检查当前用户是否具有管理员权限
$runAsAdmin = [Security.Principal.WindowsPrincipal][Security.Principal.WindowsIdentity]::GetCurrent()
$adminRole = [Security.Principal.WindowsBuiltInRole]::Administrator

if (-not $runAsAdmin.IsInRole($adminRole)) {
    # 当前用户没有管理员权限，重新以管理员身份运行脚本
    $arguments = "-NoProfile -ExecutionPolicy Bypass -File `"" + $MyInvocation.MyCommand.Definition + "`" -Wait" 
    Start-Process powershell.exe -Verb RunAs -ArgumentList $arguments
    exit
}

# 以下是需要管理员权限执行的代码
Write-Host "脚本正在以管理员权限运行..." -ForegroundColor Green

# 设置证书存放目录（请修改为你的证书文件夹路径）
# 使用脚本所在目录作为基础路径
$scriptDir = Split-Path -Parent $MyInvocation.MyCommand.Definition
$certFolder = Join-Path $scriptDir "midCAs"

# 检查目录是否存在
if (-Not (Test-Path $certFolder)) {
    Write-Host "指定的目录不存在: $certFolder" -ForegroundColor Red
    Pause
    exit 1
}

# 获取所有 .crt 和 .cer 证书文件
$certFiles = Get-ChildItem -Path $certFolder -Include "*.crt", "*.cer" -Recurse


# 如果目录中没有证书，提示并退出
if ($certFiles.Count -eq 0) {
    Write-Host "未找到任何 .crt 或 .cer 证书文件。" -ForegroundColor Yellow
    exit 0
}

# 计数器：成功安装的证书数量
$InstalledCount = 0

# 打开本地计算机的 "Intermediate Certification Authorities" 存储区
$store = New-Object System.Security.Cryptography.X509Certificates.X509Store("CA", "LocalMachine")
$store.Open("ReadWrite")

# 遍历证书文件并安装
foreach ($certFile in $certFiles) {
    try {
        Write-Host "正在安装证书: $($certFile.FullName)" -ForegroundColor Cyan

        # 读取证书
        $cert = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2
        $cert.Import($certFile.FullName)

        # 安装证书
        $store.Add($cert)
        $InstalledCount++
    }
    catch {
        Write-Host "安装失败: $($certFile.FullName) - $($_.Exception.Message)" -ForegroundColor Red
    }
}

# 关闭证书存储
$store.Close()

# 输出最终安装结果
Write-Host "证书安装完成，共成功安装 $InstalledCount 个证书。" -ForegroundColor Magenta

# 脚本执行结束后暂停并等待用户按键
Write-Host "按任意键关闭窗口..."
Read-Host
# SIG # Begin signature block
# MIIpgwYJKoZIhvcNAQcCoIIpdDCCKXACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDUfufiM4zXlwW8
# Wxig46uuNLaeXT2X2zYzzCk3KU5mbaCCDnQwggbNMIIEtaADAgECAhEAu/DMtbe4
# Mf0hrjJ3iuQMiTANBgkqhkiG9w0BAQwFADCBgDELMAkGA1UEBhMCUEwxIjAgBgNV
# BAoTGVVuaXpldG8gVGVjaG5vbG9naWVzIFMuQS4xJzAlBgNVBAsTHkNlcnR1bSBD
# ZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTEkMCIGA1UEAxMbQ2VydHVtIFRydXN0ZWQg
# TmV0d29yayBDQSAyMB4XDTIxMDUxOTA1MzIxM1oXDTM2MDUxODA1MzIxM1owajEL
# MAkGA1UEBhMCUEwxITAfBgNVBAoTGEFzc2VjbyBEYXRhIFN5c3RlbXMgUy5BLjE4
# MDYGA1UEAxMvQ2VydHVtIEV4dGVuZGVkIFZhbGlkYXRpb24gQ29kZSBTaWduaW5n
# IDIwMjEgQ0EwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCUvvjwlqjf
# 9eedRTK+NoavZlVxmQTPIN1nK5XQRUQWyPWwHD54wa2aD2kRDrAjBf71/T0XN9Oq
# ms5ML/Jolgryv3JW7/yahp7lg/wn3PDnNwzC1BlFMyCdEktQmUXMmE4mmxSutRY1
# OGGagENWPjbQ1R/x4HuTByWbG+TUV7dIPxE7TMy9Npco4f09DN6QFRXdk7bZTu58
# kxu6MFH6mzQkC3IXvExfo0c/CT4/hTZD2XknHjXLzi9jzInSHS9GQqeXd6nWbkwE
# u7znKpNEcSElorDSEcRhc1OSOLjUK0x0cOw8whzQ0fWGkPoqgCslbjZoko7U7kj/
# Fn4TWQ8s75LHjhVao+w2SWMP0ELWYIkq1ComGq3wqBD7VU6rtCDoUf237HkgRBxL
# XFjwp444oKWnchYMZ7epWX8jqntSr4QHeidL72c6XswQIIZsFglV4BO28jO02yS7
# w4iyY0c6PQs34GZ7SWphsN1DaiX9ysc6hovVYoVA94t+3p4TI1RtK9Ayh0atmfYP
# JqtBfk5kJ1h4nNax0lZmJL9chJjZQb+7XW8MQgstiQdsegaz2QL6Edm+94N8IW6T
# YaalyO9LeR35A9MoTLo79rdB04naTqWdPTBKh7DpS53ZAQn5dYs2per+m7ZeIxnE
# WaNcG+PbRh+ZslFb6M75JgN43szB+/VYVQIDAQABo4IBVTCCAVEwDwYDVR0TAQH/
# BAUwAwEB/zAdBgNVHQ4EFgQUrFfKCBbcP8UxHApN2/vx3pknLTQwHwYDVR0jBBgw
# FoAUtqFUOQLDoD+Oirz61PgcptE6Dv0wDgYDVR0PAQH/BAQDAgEGMBMGA1UdJQQM
# MAoGCCsGAQUFBwMDMDAGA1UdHwQpMCcwJaAjoCGGH2h0dHA6Ly9jcmwuY2VydHVt
# LnBsL2N0bmNhMi5jcmwwbAYIKwYBBQUHAQEEYDBeMCgGCCsGAQUFBzABhhxodHRw
# Oi8vc3ViY2Eub2NzcC1jZXJ0dW0uY29tMDIGCCsGAQUFBzAChiZodHRwOi8vcmVw
# b3NpdG9yeS5jZXJ0dW0ucGwvY3RuY2EyLmNlcjA5BgNVHSAEMjAwMC4GBFUdIAAw
# JjAkBggrBgEFBQcCARYYaHR0cDovL3d3dy5jZXJ0dW0ucGwvQ1BTMA0GCSqGSIb3
# DQEBDAUAA4ICAQC7khlZjFbU+gsRdcraFFJKj1PL63MVudnm/AILuNaxm7LIZOYt
# rZaRWEpR0qd3uhSpUEI29fMBZvuu3CHXJsM58hbJFo4YoHnyDJgTkeVmc10fd1NI
# //cI3nRD1/Alx09yWhPZlze0A/Bvwc3LcZtTywfsL5tD3QAL8ZqelrHt1Eb8AnDE
# nAzKNbm1mwYmD/5dy1XkIKfsM1pmRM2ZDcJ4qpwVewE7g+4e+Y+iq42LuaHPvNAk
# wq0daA9Hp3qemk0cRFfM5257MvEsu+6w8+44mpQUOgpASVIjY3l2vdf6BUfBJfsd
# 3pcp/2hfDnXaql9AoB7ps7eCZb9VXKm/u2lTRMf5esOcZFy4udsnGBaOwSSUe0ch
# YusfJ9IjsmpoMl4CW5E0R/U55Bz6TpDo+gaXQJL4i/zv0UR5papGsDguBASp9g/O
# fnvGTeWrS9l5orEktu9ExrFfPvAKxZK+4AO9bapPhMATraKo6kx6K51pS1ZO2pf1
# r3B+lXN5bTLgp8g/dygUZm2cmauBAzJA5v/iHOb0b4b4Muqc8dxgcX1jZRLtLvxn
# AWxkKIMfj7A4M0JrN4rLBUpPFM0+Bp2sR7rtFqNZW56huaDkxO75eNRN2z4A959R
# 5GBRnFWtgE65+B1RHAQQ/TOF4mU0Y3oIhVQr3rUh4tdSvIxr8QpsDKIvsDCCB58w
# ggWHoAMCAQICEGFf5RFemGD9MFPQVoHSloUwDQYJKoZIhvcNAQELBQAwajELMAkG
# A1UEBhMCUEwxITAfBgNVBAoTGEFzc2VjbyBEYXRhIFN5c3RlbXMgUy5BLjE4MDYG
# A1UEAxMvQ2VydHVtIEV4dGVuZGVkIFZhbGlkYXRpb24gQ29kZSBTaWduaW5nIDIw
# MjEgQ0EwHhcNMjUwMTE3MDcxMDI1WhcNMjYwMTE3MDcxMDI0WjCCASgxHTAbBgNV
# BA8TFFByaXZhdGUgT3JnYW5pemF0aW9uMRMwEQYLKwYBBAGCNzwCAQMTAkNOMRow
# GAYLKwYBBAGCNzwCAQIMCeS4iua1t+W4gjEaMBgGCysGAQQBgjc8AgEBDAnkuIrm
# tbfluIIxGzAZBgNVBAUTEjkxMzEwMTA0MDY2MDQzMTkzOTELMAkGA1UEBhMCQ04x
# EjAQBgNVBAgMCeS4iua1t+W4gjESMBAGA1UEBwwJ5LiK5rW35biCMTMwMQYDVQQK
# DCrkuprmlbDkv6Hmga/np5HmioDvvIjkuIrmtbfvvInmnInpmZDlhazlj7gxMzAx
# BgNVBAMMKuS6muaVsOS/oeaBr+enkeaKgO+8iOS4iua1t++8ieaciemZkOWFrOWP
# uDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALCsvpCjm5CCmUV3ib6d
# q2yMnT+CFCKvEb6JtHvZ7aCDUkfQlFuNG59rnk6P5NX4UVdnXwlaPNebue8Y7zRx
# U6Dsp162H/07c+cGLI+9513UsXJaoNGVm7zHTfb/hCZVBP2xgYikoQaBnQnbW9s9
# n0JL8lrR3Pet3tWPYXglPaZO2rrDXm1kB871uM4pbz1NDggmlW47PTxVAehzTkLd
# PnwG8s6AGCKu1MTbPmPirruB366ceLUCQHZsUNY2enlmPjONiBLPhuRnWvWyUbVB
# CZcM6Nadhs77/ytV+K8QWxJQK0vL/iMbBx/DzTcHrJVhKK8BpqxKX7trnAacx36B
# Y6fKLTpNF9/XFsi9+uwivtrOrMgQHeacOl2zFW7Vpg0xqZlbZv49IL41pzT3GtDP
# rNcqbykTNqIV432FEXtkoL/S8DYaktLljl4Eg0y460LCu//mtSw0ozE7yY8ZfEHR
# 63I1SjYnARo8y5H4lMhoAS7Us5i8R20CdmTQlQ9a/g0mYfi8S2LpdniRXWmpe4SX
# HZtVnVXWkYslHsc249ZhxEHQj4B0pmpRSX8tPiJ+PGVjF68muffa7zI8ZSWv1Csc
# kswKRnx2vdpF7BqiuwHpwt47coNKvTWlveR7iLzC++sZRw7sbcYyh/on0PPhbl1j
# XUEbV2iU6RhPq46mVOOqKb/JAgMBAAGjggF/MIIBezAMBgNVHRMBAf8EAjAAMEEG
# A1UdHwQ6MDgwNqA0oDKGMGh0dHA6Ly9jZXZjc2NhMjAyMS5jcmwuY2VydHVtLnBs
# L2NldmNzY2EyMDIxLmNybDB3BggrBgEFBQcBAQRrMGkwLgYIKwYBBQUHMAGGImh0
# dHA6Ly9jZXZjc2NhMjAyMS5vY3NwLWNlcnR1bS5jb20wNwYIKwYBBQUHMAKGK2h0
# dHA6Ly9yZXBvc2l0b3J5LmNlcnR1bS5wbC9jZXZjc2NhMjAyMS5jZXIwHwYDVR0j
# BBgwFoAUrFfKCBbcP8UxHApN2/vx3pknLTQwHQYDVR0OBBYEFMqP+IbE1W7lA9km
# KHwmSC86+WoMMEoGA1UdIARDMEEwBwYFZ4EMAQMwNgYLKoRoAYb2dwIFAQcwJzAl
# BggrBgEFBQcCARYZaHR0cHM6Ly93d3cuY2VydHVtLnBsL0NQUzATBgNVHSUEDDAK
# BggrBgEFBQcDAzAOBgNVHQ8BAf8EBAMCB4AwDQYJKoZIhvcNAQELBQADggIBAAUO
# XNa8c5yx0Y2eonAH3IvBFLa2foUpqDp265VCXFuR6qRtBwSqKcLkDrSDHuiWgzZZ
# ByuyvyFUyfRiO4lMLIhTD73kSc3Cw2bFP3aWIYVozTvJ1/07tUBHvGlP7KqvRsWQ
# pvi3Hw+VvXHXTIms+7FnRflU/L1sz1uknpChJWEadlN6N3B+0AUdmsMjFK23hcu/
# 84WGCQ0HaAxQ9q/7Mw2hq/SDs97xalQH0/dVNs1hwXjaDZ4iuN6jtbMDTZFUkyXS
# f6jiPVm8NrG605lG67iToVo3UFuJe29b1oS7Zz45+7MN5CKptyRqn46aYokuHVoA
# TG+eCMEhCV4I6nVapeo/6G0Bn6n1bU0ql1yn0e4viK0Qehk+QUPaTXYR7CbQO4r1
# Mp4MkiRDHR6Ipu4/exxvywbsYZXqTC8dkGCM3MBCd6mFXQowFFQc2Z56abdnkqeo
# 2DWNL86gfKYxPoaxoLdjx133qCdJyNP9LY23V6w/qxdXlUItM7z72Ly47LRsJm6A
# 3DeC7VjhHobMCv0tCDbUYkLKke7DSORf4mns3ZRO3rTXU2JAhnnZoNKxOZvH/aqR
# hOA8JhUcg4D72h6a7UeGUo2fSsCsQaiLpValoc1ndQY31Mwn3VkzWlno8X4h9Lf3
# 89bsq0SP6gYs5M2xfFR0DM/yY+tptfp7yy/eQHD5MYIaZTCCGmECAQEwfjBqMQsw
# CQYDVQQGEwJQTDEhMB8GA1UEChMYQXNzZWNvIERhdGEgU3lzdGVtcyBTLkEuMTgw
# NgYDVQQDEy9DZXJ0dW0gRXh0ZW5kZWQgVmFsaWRhdGlvbiBDb2RlIFNpZ25pbmcg
# MjAyMSBDQQIQYV/lEV6YYP0wU9BWgdKWhTANBglghkgBZQMEAgEFAKB8MBAGCisG
# AQQBgjcCAQwxAjAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQB
# gjcCAQsxDjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCCXza5CjBo0K/ZY
# xUU8l8F0hBstAXaAIRS2eGooKG+zfzANBgkqhkiG9w0BAQEFAASCAgAfee3607ea
# Fth3OJMWbYw41i18w2vqvdjb5gMT3glS8npTnUqInjfml/l3qWdNG9qeGhdQ2zUL
# o+xtNTwrS6Qk/BIOx8RNjMZKklXli1lc3Vla92Zq6mhBn4CEH2NhUGsgk+Z7bZD8
# 0Zcpa18G2stcc1I3h9f97BkXk7ebSaPbB0/OP3ymswgSk1f5lMYi8mkQfEfHehvg
# TpDYc1jQYfhcjR3+uKf5hETwEplNoiTR04VJ6b0bZ2u3vXCgmyNCW5dG2TcrfSss
# j/DreSXwv9ecqz9+0NO5XyjCGbML0vC4uyjrxOIh0w4u92p1JR4RBvBVoBp3tjFj
# MEKgcTFtSDWRMNnfTVUFSbU2YZAh56+wvE1+Jopokl/6WXqZ2sPQyITRvPyI5EMi
# St+MdEI0YEeGb7t4KIHiYmjJ2clO9uU6maraqQKvLSqzbAgGU47yiBvZM5zDsCAO
# y4dJVROm2C+M1cMCvx/VcXgeU1MdPLW9f86M33uxHeePVgLcr7fH5WdLZWW5YkSR
# 5VGDk9ueQ0qOKcx2Z2C6EZyy8dCgu/URCl8NB91an4jDEZk7tJUI4n1CqRs2YcpP
# GOS+I2n6n2iqdaW4SM+HlceK4RyuwRYMsU8ppXXTLu/A6vPz5BxRX8dyaXStWsvT
# WgLP8zs/y9OtBtXEdlXDTzsOfpXerNxvH6GCFzowghc2BgorBgEEAYI3AwMBMYIX
# JjCCFyIGCSqGSIb3DQEHAqCCFxMwghcPAgEDMQ8wDQYJYIZIAWUDBAIBBQAweAYL
# KoZIhvcNAQkQAQSgaQRnMGUCAQEGCWCGSAGG/WwHATAxMA0GCWCGSAFlAwQCAQUA
# BCDZEFmYjM+3ypJaZMEW2r1h07n1EunBIH9Hp/93k24ihAIRANcrTxfU3mqxFObB
# 6ZVVKS8YDzIwMjUwMzIxMDMzNjU0WqCCEwMwgga8MIIEpKADAgECAhALrma8Wrp/
# lYfG+ekE4zMEMA0GCSqGSIb3DQEBCwUAMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQK
# Ew5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBS
# U0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwHhcNMjQwOTI2MDAwMDAwWhcN
# MzUxMTI1MjM1OTU5WjBCMQswCQYDVQQGEwJVUzERMA8GA1UEChMIRGlnaUNlcnQx
# IDAeBgNVBAMTF0RpZ2lDZXJ0IFRpbWVzdGFtcCAyMDI0MIICIjANBgkqhkiG9w0B
# AQEFAAOCAg8AMIICCgKCAgEAvmpzn/aVIauWMLpbbeZZo7Xo/ZEfGMSIO2qZ46XB
# /QowIEMSvgjEdEZ3v4vrrTHleW1JWGErrjOL0J4L0HqVR1czSzvUQ5xF7z4IQmn7
# dHY7yijvoQ7ujm0u6yXF2v1CrzZopykD07/9fpAT4BxpT9vJoJqAsP8YuhRvflJ9
# YeHjes4fduksTHulntq9WelRWY++TFPxzZrbILRYynyEy7rS1lHQKFpXvo2GePfs
# MRhNf1F41nyEg5h7iOXv+vjX0K8RhUisfqw3TTLHj1uhS66YX2LZPxS4oaf33rp9
# HlfqSBePejlYeEdU740GKQM7SaVSH3TbBL8R6HwX9QVpGnXPlKdE4fBIn5BBFnV+
# KwPxRNUNK6lYk2y1WSKour4hJN0SMkoaNV8hyyADiX1xuTxKaXN12HgR+8WulU2d
# 6zhzXomJ2PleI9V2yfmfXSPGYanGgxzqI+ShoOGLomMd3mJt92nm7Mheng/TBeSA
# 2z4I78JpwGpTRHiT7yHqBiV2ngUIyCtd0pZ8zg3S7bk4QC4RrcnKJ3FbjyPAGogm
# oiZ33c1HG93Vp6lJ415ERcC7bFQMRbxqrMVANiav1k425zYyFMyLNyE1QulQSgDp
# W9rtvVcIH7WvG9sqYup9j8z9J1XqbBZPJ5XLln8mS8wWmdDLnBHXgYly/p1DhoQo
# 5fkCAwEAAaOCAYswggGHMA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYG
# A1UdJQEB/wQMMAoGCCsGAQUFBwMIMCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsGCWCG
# SAGG/WwHATAfBgNVHSMEGDAWgBS6FtltTYUvcyl2mi91jGogj57IbzAdBgNVHQ4E
# FgQUn1csA3cOKBWQZqVjXu5Pkh92oFswWgYDVR0fBFMwUTBPoE2gS4ZJaHR0cDov
# L2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1
# NlRpbWVTdGFtcGluZ0NBLmNybDCBkAYIKwYBBQUHAQEEgYMwgYAwJAYIKwYBBQUH
# MAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBYBggrBgEFBQcwAoZMaHR0cDov
# L2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNI
# QTI1NlRpbWVTdGFtcGluZ0NBLmNydDANBgkqhkiG9w0BAQsFAAOCAgEAPa0eH3aZ
# W+M4hBJH2UOR9hHbm04IHdEoT8/T3HuBSyZeq3jSi5GXeWP7xCKhVireKCnCs+8G
# Zl2uVYFvQe+pPTScVJeCZSsMo1JCoZN2mMew/L4tpqVNbSpWO9QGFwfMEy60HofN
# 6V51sMLMXNTLfhVqs+e8haupWiArSozyAmGH/6oMQAh078qRh6wvJNU6gnh5OruC
# P1QUAvVSu4kqVOcJVozZR5RRb/zPd++PGE3qF1P3xWvYViUJLsxtvge/mzA75oBf
# FZSbdakHJe2BVDGIGVNVjOp8sNt70+kEoMF+T6tptMUNlehSR7vM+C13v9+9ZOUK
# zfRUAYSyyEmYtsnpltD/GWX8eM70ls1V6QG/ZOB6b6Yum1HvIiulqJ1Elesj5TMH
# q8CWT/xrW7twipXTJ5/i5pkU5E16RSBAdOp12aw8IQhhA/vEbFkEiF2abhuFixUD
# obZaA0VhqAsMHOmaT3XThZDNi5U2zHKhUs5uHHdG6BoQau75KiNbh0c+hatSF+02
# kULkftARjsyEpHKsF7u5zKRbt5oK5YGwFvgc4pEVUNytmB3BpIiowOIIuDgP5M9W
# ArHYSAR16gc0dP2XdkMEP5eBsX7bf/MGN4K3HP50v/01ZHo/Z5lGLvNwQ7XHBx1y
# omzLP8lx4Q1zZKDyHcp4VQJLu2kWTsKsOqQwggauMIIElqADAgECAhAHNje3JFR8
# 2Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMjAzMjMwMDAwMDBaFw0z
# NzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1
# NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoIC
# AQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbSg9GeTKJtoLDMg/la9hGhRBVCX6SI
# 82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0hNoR8XOxs+4rgISKIhjf69o9
# xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXnHwZljZQp09nsad/ZkIdGAHvbREGJ
# 3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0VAshaG43IbtArF+y3kp9zvU5Emfv
# DqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4fsbVYTXn+149zk6wsOeKlSNbwsDET
# qVcplicu9Yemj052FVUmcJgmf6AaRyBD40NjgHt1biclkJg6OBGz9vae5jtb7IHe
# IhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0QCirc0PO30qhHGs4xSnzyqqWc0Jo
# n7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+DrhkKvp1KCRB7UK/BZxmSVJQ
# 9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA+bIwpUzX6ZhKWD7TA4j+s4/T
# Xkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk42PgpuE+9sJ0sj8eCXbsq11GdeJg
# o1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzSM7TNsQIDAQABo4IBXTCCAVkw
# EgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUuhbZbU2FL3MpdpovdYxqII+e
# yG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsGAQUFBwEBBGswaTAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAChjVodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNy
# dDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGln
# aUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglg
# hkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIBAH1ZjsCTtm+YqUQiAX5m1tghQuGw
# GC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8dB+k+YMjYC+VcW9dth/qEICU0
# MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIlzpVpP0d3+3J0FNf/q0+KLHqrhc1D
# X+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp876i8dU+6WvepELJd6f8oVInw
# 1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2nrF5mYGjVoarCkXJ38SNoOeY
# +/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3ZpHxcpzpSwJSpzd+k1OsOx0I
# SQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJshIUDQtxMkzdwdeDrknq3lNHGS1yZr
# 5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6OOmc4d0j/R0o08f56PGYX/sr2H7y
# Rp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDwN7+YAN8gFk8n+2BnFqFmut1VwDop
# hrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZvAT6gt4y3wSJ8ADNXcL50CN/
# AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQrH4D6wPIOK+XW+6kvRBVK5xMO
# Hds3OBqhK/bt1nz8MIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkq
# hkiG9w0BAQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5j
# MRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBB
# c3N1cmVkIElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5
# WjBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQL
# ExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJv
# b3QgRzQwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1K
# PDAiMGkz7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2r
# snnyyhHS5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C
# 8weE5nQ7bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBf
# sXpm7nfISKhmV1efVFiODCu3T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGY
# QJB5w3jHtrHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8
# rhsDdV14Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaY
# dj1ZXUJ2h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+
# wJS00mFt6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw
# ++hkpjPRiQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+N
# P8m800ERElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7F
# wI+isX4KJpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUw
# AwEB/zAdBgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAU
# Reuir/SSy4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEB
# BG0wazAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsG
# AQUFBzAChjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1
# cmVkSURSb290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRp
# Z2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAow
# CDAGBgRVHSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/
# Vwe9mqyhhyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLe
# JLxSA8hO0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE
# 1Od/6Fmo8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY9Hda
# XFSMb++hUD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbO
# byMt9H5xaiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMYID
# djCCA3ICAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIElu
# Yy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYg
# VGltZVN0YW1waW5nIENBAhALrma8Wrp/lYfG+ekE4zMEMA0GCWCGSAFlAwQCAQUA
# oIHRMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcN
# MjUwMzIxMDMzNjU0WjArBgsqhkiG9w0BCRACDDEcMBowGDAWBBTb04XuYtvSPnvk
# 9nFIUIck1YZbRTAvBgkqhkiG9w0BCQQxIgQg3XqSHJ+THV8NCRet3pPUjA3n4ByS
# 5/6nzuQAAQ2aH38wNwYLKoZIhvcNAQkQAi8xKDAmMCQwIgQgdnafqPJjLx9DCzoj
# MK7WVnX+13PbBdZluQWTmEOPmtswDQYJKoZIhvcNAQEBBQAEggIAFrK9foeHRt7N
# +nAhQC4aXtP0LHbV6YzcCgYla6pUBjT1Bx2vHS5digwwElMmP/3n+WypBPpUPtFA
# hOqJPmQTkY44+DJWcTdLEx2CzRAP8O7vsGbdfP2keqnlpF+ca6UEwrI9zOfPK2Sx
# 1396gVcldzJcPQKEPPy5GHOu/Epjzg+8Bz43G13lgjQF7CbfF8kbq64UTb+CE1em
# uNPsS1XnS+zVGyBpzfy6PmAV8MC8eYXt4l47xd5nYsMYXUZb+hbHrQj6pGKWRT1J
# /Jaf90+UVx4n+RI5KP3P+UQ38dgpJ6B5FXKnKOly5ecs5aXmKYF+4zGDLzEpxjQY
# CEPgTj9XU8zJ88JqtFjAnD005uvexsvTY4hJVdFcngkmnILDO8fSd7jIRTF1Qkrj
# WM/t7R2Or61dfCS5gedE7qMiW1w8+vafqvm5jL2BZyK5zZJaB8N+ugVC/Xvc5mCK
# 9pbRb4qBttelyO3REVDKCb1yY49LvKYOnffgRlDV0tOQw77KrczJyNjP6b9P1bO2
# X70lC+m/dG7wJ8xGVjY5W31cA5R2F+byV2T/dg9IwzHjgndDWCCiHeq7DJeelLFt
# bPpRA5hMS3gGHBKyG0eJHhh9C067kYu4maxrKEm72tlwfkb9K4a5lcmdoyaqW/go
# 8wPd2ZutsEOaqmiFC3iheukqN/Y96pI=
# SIG # End signature block
